// Content Repurpose Chrome Extension
// Production API endpoint
const DEFAULT_API_ENDPOINT = 'https://content-repurpose.pages.dev';

let apiEndpoint = DEFAULT_API_ENDPOINT;
let currentUrl = '';
let currentTitle = '';

// Initialize popup
document.addEventListener('DOMContentLoaded', async () => {
    // Load saved settings
    const settings = await chrome.storage.sync.get(['apiEndpoint']);
    if (settings.apiEndpoint) {
        apiEndpoint = settings.apiEndpoint;
        document.getElementById('api-endpoint').value = apiEndpoint;
    }
    
    // Update dashboard link
    document.getElementById('dashboard-link').href = apiEndpoint;
    
    // Get current tab info
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (tab) {
        currentUrl = tab.url;
        currentTitle = tab.title;
        
        document.getElementById('page-url').textContent = currentUrl;
        document.getElementById('page-title').textContent = currentTitle;
        
        // Auto-detect YouTube
        if (currentUrl.includes('youtube.com') || currentUrl.includes('youtu.be')) {
            document.getElementById('page-title').innerHTML = `🎥 ${currentTitle}`;
        }
    }
    
    // Platform button toggles
    document.querySelectorAll('.platform-btn').forEach(btn => {
        btn.addEventListener('click', () => {
            const checkbox = btn.querySelector('input');
            checkbox.checked = !checkbox.checked;
            btn.classList.toggle('active', checkbox.checked);
        });
    });
    
    // Settings toggle
    document.getElementById('settings-toggle').addEventListener('click', () => {
        document.getElementById('settings-panel').classList.toggle('show');
    });
    
    // Save settings
    document.getElementById('save-settings').addEventListener('click', async () => {
        const newEndpoint = document.getElementById('api-endpoint').value.trim();
        if (newEndpoint) {
            apiEndpoint = newEndpoint.replace(/\/$/, ''); // Remove trailing slash
            await chrome.storage.sync.set({ apiEndpoint });
            document.getElementById('dashboard-link').href = apiEndpoint;
            showStatus('Settings saved!', 'success');
        }
    });
    
    // Save button
    document.getElementById('save-btn').addEventListener('click', saveToAirtable);
});

async function saveToAirtable() {
    const saveBtn = document.getElementById('save-btn');
    const customTitle = document.getElementById('custom-title').value.trim();
    const notes = document.getElementById('notes').value.trim();
    const platforms = Array.from(document.querySelectorAll('.platform-btn input:checked'))
        .map(cb => cb.value);
    
    // Disable button and show loading
    saveBtn.disabled = true;
    saveBtn.innerHTML = '<div class="spinner"></div> Saving...';
    showStatus('Saving to pipeline...', 'loading');
    
    try {
        const response = await fetch(`${apiEndpoint}/api/save`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                url: currentUrl,
                title: customTitle || currentTitle,
                notes: notes,
                platforms: platforms
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            showStatus('✅ URL saved to pipeline!', 'success');
            saveBtn.innerHTML = '✅ Saved!';
            
            // Reset after 2 seconds
            setTimeout(() => {
                saveBtn.innerHTML = '<span>💾</span> Save to Pipeline';
                saveBtn.disabled = false;
            }, 2000);
        } else {
            throw new Error(result.error || 'Failed to save');
        }
    } catch (error) {
        console.error('Error:', error);
        showStatus(`❌ ${error.message}`, 'error');
        saveBtn.innerHTML = '<span>💾</span> Save to Pipeline';
        saveBtn.disabled = false;
    }
}

function showStatus(message, type) {
    const status = document.getElementById('status');
    status.textContent = message;
    status.className = `status ${type}`;
}
